'use client';

import { useState, useEffect } from 'react';
import { useTranslations } from 'next-intl';
import Header from '../components/Header';
import Footer from '../components/Footer';
import AnimatedText from '@/components/AnimatedText';
import AnimatedElement from '@/components/AnimatedElement';
import Image from 'next/image';

export default function LakeTokenSecurityPage() {
  const t = useTranslations('LakeTokenSecurityPage');
  const [isDesktop, setIsDesktop] = useState(false);

  useEffect(() => {
    const checkScreenSize = () => {
      setIsDesktop(window.innerWidth >= 1024);
    };

    checkScreenSize();
    window.addEventListener('resize', checkScreenSize);

    return () => window.removeEventListener('resize', checkScreenSize);
  }, []);

  const securityFeatures = [
    {
      title: t('security_features.contract_features.title'),
      subtitle: t('security_features.contract_features.subtitle'),
      description: t('security_features.contract_features.description'),
      code: t.raw('security_features.contract_features.functions'),
      image: "/images/lake-token/security/contract-features.png"
    },
    {
      title: t('security_features.fixed_supply.title'),
      description: t('security_features.fixed_supply.description'),
      benefit: t('security_features.fixed_supply.benefit'),
      benefitIcon: "/images/lake-token/security/1.svg",
      image: "/images/lake-token/security/fixed-supply.png"
    },
    {
      title: t('security_features.no_ownership.title'),
      description: t('security_features.no_ownership.description'),
      benefit: t('security_features.no_ownership.benefit'),
      benefitIcon: "/images/lake-token/security/2.svg",
      image: "/images/lake-token/security/no-ownership.png"
    },
    {
      title: t('security_features.no_external_calls.title'),
      description: t('security_features.no_external_calls.description'),
      benefit: t('security_features.no_external_calls.benefit'),
      benefitIcon: "/images/lake-token/security/3.svg",
      image: "/images/lake-token/security/no-external-calls.png"
    },
    {
      title: t('security_features.safe_arithmetic.title'),
      description: t('security_features.safe_arithmetic.description'),
      benefit: t('security_features.safe_arithmetic.benefit'),
      benefitIcon: "/images/lake-token/security/4.svg",
      image: "/images/lake-token/security/safe-arithmetic.png"
    },
    {
      title: t('security_features.gas_optimizations.title'),
      description: t('security_features.gas_optimizations.description'),
      benefit: t('security_features.gas_optimizations.benefit'),
      benefitIcon: "/images/lake-token/security/5.svg",
      image: "/images/lake-token/security/gas-optimizations.png"
    },
    {
      title: t('security_features.verified.title'),
      description: t('security_features.verified.description'),
      benefit: t('security_features.verified.benefit'),
      benefitIcon: "/images/lake-token/security/6.svg",
      image: "/images/lake-token/security/verified.png"
    }
  ];

  const securityChecks = [
    { check: t('security_checks.backdoors'), status: t('security_checks.none') },
    { check: t('security_checks.hidden_mint_burn'), status: t('security_checks.none') },
    { check: t('security_checks.blacklisting'), status: t('security_checks.none') },
    { check: t('security_checks.pausing'), status: t('security_checks.none') },
    { check: t('security_checks.fees'), status: t('security_checks.none') },
    { check: t('security_checks.external_calls'), status: t('security_checks.none') },
    { check: t('security_checks.upgradeable_proxy'), status: t('security_checks.none') }
  ];

  if (isDesktop) {
    return (
      <div className="min-h-screen bg-[#EFEFEF] flex flex-col">
        <Header />
        <main className="flex-1 w-full flex flex-col items-center">
          {/* Security Summary Header with Check Status Table */}
          <div className="w-[1140px] flex flex-col gap-8 mb-16">
            {/* Header with icon and title */}
            <div className="w-full flex justify-center">
              <div className="flex items-center gap-4 min-h-[65px]">
                <div className="w-[65px] h-[65px] flex items-center justify-center">
                  <Image src="/images/lake-token/security/Mask groupSec.svg" alt="Security" width={65} height={65} />
                </div>
                <AnimatedText
                  as="h1"
                  animation="slideInUp"
                  delay={200}
                  className="text-[#002D64] text-[55.067px] font-bold font-['Inter'] leading-[66.08px] m-0"
                >
                  {t('title')}
                </AnimatedText>
              </div>
            </div>

            {/* Check Status Table */}
            <div className="flex flex-col gap-8">
              <div className="flex justify-between">
                <AnimatedText
                  as="h2"
                  animation="slideInLeft"
                  delay={300}
                  className="text-[#1C1C1C] text-2xl font-bold font-['Inter']"
                >
                  {t('check')}
                </AnimatedText>
                <AnimatedText
                  as="h3"
                  animation="slideInLeft"
                  delay={400}
                  className="text-[#1C1C1C] text-2xl font-bold font-['Inter']"
                >
                  {t('status')}
                </AnimatedText>
              </div>
              <div className="h-px bg-[#002D64]" />
              
              {securityChecks.map((item, index) => (
                <div key={index} className="space-y-2">
                  <div className="flex justify-between">
                    <AnimatedText
                      as="p"
                      animation="slideInLeft"
                      delay={500 + index * 100}
                      className="text-[#1C1C1C] text-lg font-normal font-['Inter']"
                    >
                      {item.check}
                    </AnimatedText>
                    <AnimatedText
                      as="p"
                      animation="slideInLeft"
                      delay={550 + index * 100}
                      className="text-[#1C1C1C] text-lg font-normal font-['Inter']"
                    >
                      {item.status}
                    </AnimatedText>
                  </div>
                  <div className="h-px bg-[#002D64]" />
                </div>
              ))}
            </div>
          </div>

          {/* Security Features */}
          <div className="w-full space-y-0">
            {securityFeatures.map((feature, index) => (
              <AnimatedElement
                key={index}
                animation="slideInUp"
                delay={200 + index * 200}
                className="w-full h-[836px] bg-[#EFEFEF] relative"
              >
                <div className="w-full h-[142px] absolute top-0 bg-gradient-to-b from-[rgba(217,217,217,0.5)] to-transparent" />
                
                <div className="w-full h-full flex items-center justify-center">
                  <div className="w-[1140px] flex items-center gap-16">
                    {/* Image */}
                    <div className="w-[534px] h-[534px] rounded-[50px] overflow-hidden bg-gray-300 flex-shrink-0">
                      <Image 
                        src={feature.image}
                        alt={feature.title}
                        width={534}
                        height={534}
                        className="w-full h-full object-cover"
                      />
                    </div>

                    {/* Content */}
                    <div className="flex-1">
                      <AnimatedText
                        as="h2"
                        animation="slideInRight"
                        delay={300 + index * 200}
                        className="text-[#002D64] text-[45px] font-bold font-['Inter'] leading-[54px] mb-6"
                      >
                        {feature.title}
                      </AnimatedText>
                      
                      {feature.subtitle && (
                        <AnimatedText
                          as="h3"
                          animation="slideInRight"
                          delay={400 + index * 200}
                          className="text-[#1C1C1C] text-[25px] font-medium font-['Inter'] leading-[30px] mb-6"
                        >
                          {feature.subtitle}
                        </AnimatedText>
                      )}

                      <div className="mb-6">
                        {feature.description.split('\n').map((line, lineIndex) => (
                          <AnimatedText
                            key={lineIndex}
                            as="p"
                            animation="slideInRight"
                            delay={500 + index * 200 + lineIndex * 50}
                            className="text-[#1C1C1C] text-lg font-normal font-['Inter'] flex items-start gap-2"
                          >
                            <span className="text-[#002D64] font-bold mt-2 flex-shrink-0">•</span>
                            <span>{line}</span>
                          </AnimatedText>
                        ))}
                      </div>

                      {feature.code && Array.isArray(feature.code) && (
                        <div className="mb-6">
                          {feature.code.map((line: string, codeIndex: number) => (
                            <AnimatedText
                              key={codeIndex}
                              as="p"
                              animation="slideInRight"
                              delay={600 + index * 200 + codeIndex * 50}
                              className="text-[#1C1C1C] text-lg font-normal font-['Inter'] underline"
                            >
                              {line}
                            </AnimatedText>
                          ))}
                        </div>
                      )}

                      {feature.benefit && (
                        <div className="flex items-center gap-3">
                          <div className="w-6 h-6 flex items-center justify-center">
                            <Image src={feature.benefitIcon} alt="benefit" width={24} height={24} />
                          </div>
                          <AnimatedText
                            as="p"
                            animation="slideInRight"
                            delay={700 + index * 200}
                            className="text-[#002D64] text-lg font-bold font-['Inter']"
                          >
                            {feature.benefit}
                          </AnimatedText>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              </AnimatedElement>
            ))}
          </div>
        </main>
        <Footer />
      </div>
    );
  }

  // Mobile version
  return (
    <div className="min-h-screen bg-[#EFEFEF] flex flex-col">
      <Header />
      <main className="flex-1 w-full flex flex-col py-8">
        {/* Security Summary Header */}
        <div className="w-full flex justify-center mb-8">
          <div className="flex items-center gap-4 min-h-[65px]">
            <div className="w-[65px] h-[65px] flex items-center justify-center">
              <Image src="/images/lake-token/security/Mask groupSec.svg" alt="Security" width={65} height={65} />
            </div>
            <AnimatedText
              as="h1"
              animation="slideInUp"
              delay={200}
              className="text-[#002D64] text-3xl font-bold font-['Inter'] leading-tight m-0"
            >
              {t('title')}
            </AnimatedText>
          </div>
        </div>

        {/* Security Checks */}
        <div className="mb-8 mx-3">
          <AnimatedText
            as="h2"
            animation="slideInLeft"
            delay={300}
            className="text-[#1C1C1C] text-xl font-bold font-['Inter'] mb-4"
          >
            {t('check_status')}
          </AnimatedText>
          
          <div className="space-y-4">
            {securityChecks.map((item, index) => (
              <div key={index} className="flex justify-between items-center py-2 border-b border-[#002D64]">
                <AnimatedText
                  as="p"
                  animation="slideInLeft"
                  delay={400 + index * 100}
                  className="text-[#1C1C1C] text-base font-normal font-['Inter']"
                >
                  {item.check}
                </AnimatedText>
                <AnimatedText
                  as="p"
                  animation="slideInRight"
                  delay={450 + index * 100}
                  className="text-[#1C1C1C] text-base font-normal font-['Inter']"
                >
                  {item.status}
                </AnimatedText>
              </div>
            ))}
          </div>
        </div>

        {/* Security Features */}
        <div className="space-y-8">
          {securityFeatures.map((feature, index) => (
            <AnimatedElement
              key={index}
              animation="slideInUp"
              delay={200 + index * 200}
              className="bg-[#EFEFEF] rounded-lg p-3 shadow-lg mx-3"
            >
              <div className="w-full h-64 rounded-lg overflow-hidden bg-gray-300 mb-6">
                <Image 
                  src={feature.image}
                  alt={feature.title}
                  width={400}
                  height={256}
                  className="w-full h-full object-cover"
                />
              </div>

              <AnimatedText
                as="h2"
                animation="slideInUp"
                delay={300 + index * 200}
                className="text-[#002D64] text-2xl font-bold font-['Inter'] leading-tight mb-4"
              >
                {feature.title}
              </AnimatedText>
              
              {feature.subtitle && (
                <AnimatedText
                  as="h3"
                  animation="slideInUp"
                  delay={400 + index * 200}
                  className="text-[#1C1C1C] text-lg font-medium font-['Inter'] leading-relaxed mb-4"
                >
                  {feature.subtitle}
                </AnimatedText>
              )}

              <div className="mb-4">
                {feature.description.split('\n').map((line, lineIndex) => (
                  <AnimatedText
                    key={lineIndex}
                    as="p"
                    animation="slideInUp"
                    delay={500 + index * 200 + lineIndex * 50}
                    className="text-[#1C1C1C] text-base font-normal font-['Inter'] flex items-start gap-2"
                  >
                    <span className="text-[#002D64] font-bold mt-1 flex-shrink-0">•</span>
                    <span>{line}</span>
                  </AnimatedText>
                ))}
              </div>

              {feature.code && Array.isArray(feature.code) && (
                <div className="mb-4">
                  {feature.code.map((line: string, codeIndex: number) => (
                    <AnimatedText
                      key={codeIndex}
                      as="p"
                      animation="slideInUp"
                      delay={600 + index * 200 + codeIndex * 50}
                      className="text-[#1C1C1C] text-sm font-normal font-['Inter'] underline"
                    >
                      {line}
                    </AnimatedText>
                  ))}
                </div>
              )}

              {feature.benefit && (
                <div className="flex items-center gap-3">
                  <div className="w-6 h-6 flex items-center justify-center">
                    <Image src={feature.benefitIcon} alt="benefit" width={30} height={30} />
                  </div>
                  <AnimatedText
                    as="p"
                    animation="slideInUp"
                    delay={700 + index * 200}
                    className="text-[#002D64] text-base font-bold font-['Inter']"
                  >
                    {feature.benefit}
                  </AnimatedText>
                </div>
              )}
            </AnimatedElement>
          ))}
        </div>
      </main>
      <Footer />
    </div>
  );
} 